﻿"""This is an implementation of a driving transport, simple with forward/backward
movement and left and right turning. Like driving a car, turning only works if
moving forward/backward.
"""

import math

import viz
import vizmat

from transportation import PhysicsTransport


class Driving(PhysicsTransport):
	"""Transport class for an abstraction of simple driving."""
	def __init__(self,
					height=0,
					maxTurnAngle=90,
					steeringSensitivity=1.0,
					acceleration=4.0, # in meters per second per second, lower accelerations can be obtained by using a smaller mag on the input, e.g. pressing the joystick lower
					maxSpeed=1.44, # in meters per second, as a reference 1.4m/s is a typical walking speed, 10.44 is a very fast run
					autoBreakingDragCoef=0.1, # determines how quickly the walking transport will stop 
					dragCoef=0.0001,
					usingPhysics=False,
					**kwargs):
		
		# init the base class
		PhysicsTransport.__init__(self, **kwargs)
		
		# set some vars
		self._maxTurnAngle = maxTurnAngle
		self._steeringSensitivity = steeringSensitivity
		
		self._acceleration = acceleration
		self._maxSpeed = maxSpeed
		self._autoBreakingDragCoef = autoBreakingDragCoef
		self._dragCoef = dragCoef
		
		self._usingPhysics = usingPhysics
		
		# check if we're using physics and if so add a collision object
		if usingPhysics:
			self._initPhysics()
		
		self.setPosition(0, height, 0)
		
		self.steeringDirection = 0
		self._Vp = vizmat.Vector([0, 0, 0])
		self._Ap = vizmat.Vector([0, 0, 0])
		self._Vr = vizmat.Vector([0, 0, 0])
	
	def setVelocity(self, vel):
		"""Set velocity"""
		self._Vp = vizmat.Vector(vel)
	
	def setAngularVelocity(self, vel):
		"""Set angular velocity"""
		self._Vr = vizmat.Vector(vel)
	
	def moveForward(self, mag=1):
		"""Move forward"""
		self._Ap[2] = mag
	
	def moveBackward(self, mag=1):
		"""Move backward"""
		self._Ap[2] = -mag
	
	def turnLeft(self, mag=1):
		"""Turn left"""
		self._Vr[0] -= mag
	
	def turnRight(self, mag=1):
		"""Turn right"""
		self._Vr[0] += mag
	
	def getAccelerationRotationMatrix(self):
		"""Returns the acceleration rotation matrix, i.e. the matrix used to
		determine reference frame for acceleration.
		
		@return vizmat.Transform()
		"""
		rotMat = vizmat.Transform()
		rotMat.setEuler([self.getEuler()[0], 0, 0])
		return rotMat
	
	def finalize(self):
		"""Function which executes the quequed functions such as
		moveForward and moveBack basing them off the sample orientation
		from the tracker. Should be called regularly either by a timer
		or ideally at every frame.
		"""
		
		super(Driving, self).finalize()
		self._updateTime()
		idt = min(60.0, 1.0/self._dt)
		
		# if necessary normalize the acceleration
		# not necessary here because we've only got linear acceleration, so just bind it
		self._Ap[2] = min(1, max(-1, self._Ap[2]))
		
		# scale acceleration (right now no units just 0-1 range magnitude vector)
		self._Ap[2] *= self._acceleration
		
		# get the current position
		pos = self.getPosition()
		euler = self.getEuler()
		
		# adjust the euler angle by adding on the rotation
		self.steeringDirection = min(self._maxTurnAngle, max(-self._maxTurnAngle, self.steeringDirection+self._Vr[0]*self._steeringSensitivity*self._maxTurnAngle))
		if self._Vr[0] == 0:
			self.steeringDirection = 0
		elif self._Ap[2] != 0:# only turn when driving
			euler[0] += self.steeringDirection*self._dt*math.copysign(1, self._Ap[2])
		
		# scale acceleration (right now no units just 0-1 range magnitude vector)
		self._Ap *= self._acceleration
		
		rotMat = self.getAccelerationRotationMatrix()
		
		invMat = rotMat.inverse()
		
		# we want to have a fast deceleration if we're not moving in a particular direction
		breakingVec = vizmat.Vector(invMat.preMultVec(self._Vp)) * self._autoBreakingDragCoef * idt
		localVp = invMat.preMultVec(self._Vp)
		if self._Ap[0] != 0 and (self._Ap[0]*localVp[0] > 0):
			breakingVec[0] = 0
		if self._Ap[1] != 0 and (self._Ap[1]*localVp[1] > 0):
			breakingVec[1] = 0
		if self._Ap[2] != 0 and (self._Ap[2]*localVp[2] > 0):
			breakingVec[2] = 0
		breakingVec = rotMat.preMultVec(breakingVec)
		
		# now apply the acceleration to the velocity
		drag = self._Vp * self._dragCoef * idt
		adjAp = rotMat.preMultVec(self._Ap)
		
		self._Vp[0] += (adjAp[0] - drag[0] - breakingVec[0]) * self._dt
		self._Vp[1] += (adjAp[1] - drag[1] - breakingVec[1]) * self._dt
		self._Vp[2] += (adjAp[2] - drag[2] - breakingVec[2]) * self._dt
		velMag = self._Vp.length()
		if velMag > self._maxSpeed:
			self._Vp = (self._Vp / velMag) * self._maxSpeed
		
		# apply the final position update, adjusting for the pivot if necessary
		if self._usingPhysics:
			self._physNode.setVelocity(self._Vp)
			self._physManualEuler = euler
			# zero out velocity and angular velocity so that we can use dynamic collisions
			# without using the added forces from the physics engine. Note that to use
			# collisions the update priority has to be before the physics update priority.
			self._physNode.setAngularVelocity([0, 0, 0])
		else:
			# scale the final velocity by the movement speed
			# apply the velocity to the position
			local = invMat.preMultVec(self._Vp)
			local[0] = self._movementSpeed[0]*local[0] * self._dt
			local[1] = self._movementSpeed[1]*local[1] * self._dt
			local[2] = self._movementSpeed[2]*local[2] * self._dt
			finalVp = rotMat.preMultVec(local)
			pos[0] += finalVp[0]
			pos[1] += finalVp[1]
			pos[2] += finalVp[2]
			
			if self._pivot is None:
				self.setPosition(pos)
				self.setEuler(euler)
			else:
				self._adjustForPivot(pos, euler)
		
		self._Ap[0] = 0
		self._Ap[1] = 0
		self._Ap[2] = 0
		
		self._Vr[0] = 0
	
	def _initPhyiscsNode(self):
		"""Initializing physics node"""
		physNode = viz.addGroup()
		physNode.enable(viz.COLLIDE_NOTIFY)
		# add a collide sphere to the default physics node
		physicsHeightOffset = 0.5
		pos = self._node.getPosition()
		pos[1] += physicsHeightOffset*2.0
		physNode.setPosition(pos)
		collideSphere = physNode.collideBox([1.0, physicsHeightOffset, 1.0])
		collideSphere.setBounce(0.0001)
		collideSphere.setFriction(0.00)
		self._physNode = physNode
		return physNode
	
	def _postPhysics(self):
		"""Update that happens after physics is run """
		self._physNode.setEuler(self._physManualEuler)
		self._Vp = vizmat.Vector(self._physNode.getVelocity())
		self._physNode.setAngularVelocity([0, 0, 0])
		# save matrix to compare with pre-physics matrix of the node on next frame
		self._postPhysMat = self.getMatrix()
